/*
 *  Copyright (c) 1992 John E. Davis  (davis@amy.tch.harvard.edu)
 *  All Rights Reserved.
 */

#include <stdio.h>
#include "sysdep.h"

#include <signal.h>
#include <stdlib.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <unistd.h>
#include <termios.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>

#define TTY_DESCR 2

     struct termios OLDTTY;
     /* this next works on ultrix for setting termios */
#ifdef TCGETS
#define GET_TERMIOS(fd, x) ioctl(fd, TCGETS, x)
#define SET_TERMIOS(fd, x) ioctl(fd, TCSETS, x)
#else
#define GET_TERMIOS(fd, x) tcgetattr(fd, x)
#define SET_TERMIOS(fd, x) tcsetattr(fd, TCSANOW, x)
#endif

void init_tty()
{
   struct termios newtty;

   fputs("\033=\033[?1l", stdout);   /* application keys/cursor keys */
   GET_TERMIOS(TTY_DESCR, &OLDTTY);
   GET_TERMIOS(TTY_DESCR, &newtty);
   newtty.c_iflag &= ~(ECHO);
   newtty.c_iflag &= ~(INLCR);
   newtty.c_iflag &= ~(ICRNL);
   newtty.c_cc[VMIN] = 1;
   newtty.c_cc[VTIME] = 0;
   newtty.c_cc[VEOF] = 1;
   newtty.c_lflag = ISIG;
   newtty.c_cc[VINTR] = 7;   /* ^G */
   newtty.c_cc[VQUIT] = 255;
   newtty.c_cc[VSUSP] = 255;   /* to ignore ^Z */
   SET_TERMIOS(TTY_DESCR, &newtty);
}

void reset_tty()
{
   fputs("\033>", stdout);  /* Enable the Numeric Keypad */
   SET_TERMIOS(TTY_DESCR, &OLDTTY);
}

unsigned char sys_getkey()
{
   unsigned char c;
   if (KeyBoard_Quit) return(7);
   while (!KeyBoard_Quit && !input_pending(2, 1));
   /* sleep for 1 second and try again */
   if (KeyBoard_Quit) return(7);
   if ((read(TTY_DESCR, &c, 1) < 0) && !KeyBoard_Quit)
     {
	fprintf(stderr,"getkey():  read failed.\n");
	reset_tty();
	exit(1);
     }
   /* only way for keyboard quit to be non zero is if ^G recived and sigint processed */
   if (KeyBoard_Quit) c = 7;
   KeyBoard_Quit = 0;

   return(c);
}

int input_pending(int fd, int secs)
{
    struct timeval wait;
    int readfd, writefd, exceptfd, ret;
    long usecs = 0;
    /* extern int select(int, int *, int *, int *, struct timeval *); */

    if (INPUT_BUFFER_LEN) return(1);
    if (secs == 0) usecs = 10000;  /* 1/100th of a sec */
    wait.tv_sec = secs;
    wait.tv_usec = usecs;

    readfd = 1 << fd;
    writefd = exceptfd = 0;

    ret = select(fd + 1, &readfd, &writefd, &exceptfd, &wait);
    return(ret);
}

/*  This is to get the size of the terminal  */
void get_term_dimensions(int *cols, int *rows)
{
#ifdef TIOCGWINSZ
   struct winsize wind_struct;

   ioctl(2,TIOCGWINSZ,&wind_struct);
   *cols = (int) wind_struct.ws_col;
   *rows = (int) wind_struct.ws_row;
   if (*rows <= 0) *rows = 24;
   if (*cols <= 0) *cols = 80;
#else
   *rows = 24;
   *cols = 80;
#endif
}

/* returns 0 on failure, 1 on sucess */
int sys_delete_file(char *filename)
{
    return(1 + unlink(filename));
}

int sys_rename(char *from, char *to)
{
    return(rename(from,to));
}

/* this routine returns a Static pointer which is considered volitile */
char *expand_filename(char *file)
{
   char *p, *p1, *mark, *last_mark, slash = '/', *home;
   static char work[500];

   /* This converts ~/ to $HOME/ */
   int n;

   strcpy(work, get_cwd()); strcat(work, file);
   file = work;

   p = file;
   while(*p) if (*p++ == '~') break;

   if (*p == '/')
     {
	/* at some point, I need to make this more generic for other users */
	if (NULL != (home = getenv("HOME")))
	  {
	     strcpy(work, home);
	     n = strlen(home) - 1;
	     if (home[n] == '/') work[n] = 0;
	     strcat(work, p);
	     strcpy(file, work);
	  }
     }

   /* remove ../ and ./ stuff */

    p = p1 = last_mark = mark = file;

    while(*p != 0)
      {
	  if ((*p == '.') && (*(p + 1) == '.'))
	    {
		p += 2;
		if (*p == slash)
		  {
		      mark = p1 = last_mark;
		  }
		else
		  {
		      *p1++ = '.'; *p1++ = '.'; *p1++ = *p;
		  }

	    }
	  else if ((*p == '.') && (*(p + 1) == slash))  /* do nothing */
	    {
		p++;
	    }
	  else if ((*p == slash) && (*(p + 1) == slash))
	    {
		/* // is a signal that we go back to root */
		last_mark = p1 = file;
		*p1++ = slash;
		mark = p1;
		p++;
	    }
	  else if (*p == slash)
	    {
		last_mark = mark;
		*p1++ = slash;
		mark = p1;
	    }
	  else (*p1++ = *p);

	  if (*p != 0) p++;
      }
   *p1 = 0;
   /* if (mark - file < 0) msg_error("Negative value in expand_filename"); */
   /* return(mark); */
   return(work);
}

void sys_suspend(void)
{
#ifdef _POSIX_SOURCE
   kill(getpgrp(),SIGTSTP);
#else 
#ifdef sgi
   kill(getpgrp(),SIGTSTP);
#else
   kill(getpgrp(0),SIGTSTP);
#endif
#endif
   /* According to man pages, sys5 takes no argument for getpgrp but BSD does */
}

/* returns 0 if file does not exist, 1 if it is not a dir, 2 if it is */
int sys_chmod(char *file, int what, int *mode)
{
   struct stat buf;
   int m;

   if (what)
     {
	chmod(file, *mode);
	return(0);
     }

   if (stat(file, &buf) < 0) switch (errno)
     {
	case EACCES: return(-1); /* es = "Access denied."; break; */
	case ENOENT: return(0);  /* ms = "File does not exist."; */
	case ENOTDIR: return(-2); /* es = "Invalid Path."; */
	default: return(-3); /* "stat: unknown error."; break;*/
     }

   m = buf.st_mode;

/* AIX requires this */
#ifdef _S_IFDIR
#ifndef S_IFDIR
#define S_IFDIR _S_IFDIR
#endif
#endif

   *mode = m & 0777;

   if (m & S_IFDIR) return (2);
   return(1);
}

unsigned long sys_file_mod_time(char *file)
{
   struct stat buf;

   if (stat(file, &buf) < 0) return(0);
   return((unsigned long) buf.st_mtime);
}

